// ----------------------------------------------------------------------------------
// Microsoft Developer & Platform Evangelism
// 
// Copyright (c) Microsoft Corporation. All rights reserved.
// 
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
// ----------------------------------------------------------------------------------
// The example companies, organizations, products, domain names,
// e-mail addresses, logos, people, places, and events depicted
// herein are fictitious.  No association with any real company,
// organization, product, domain name, email address, logo, person,
// places, or events is intended or should be inferred.
// ----------------------------------------------------------------------------------

namespace Client
{
    using System;
    using System.Collections.Specialized;
    using System.IdentityModel.Tokens;
    using System.Linq;
    using System.Net;
    using System.ServiceModel;
    using System.ServiceModel.Security;
    using System.ServiceModel.Web;
    using System.Text;
    using System.Web;
    using Microsoft.IdentityModel.Protocols.WSTrust;
    using Microsoft.IdentityModel.Protocols.WSTrust.Bindings;
    using ProAzure.MachineInfo.Service;

    public class Program
    {
        private const string ServiceNamespace = "proazure-1";
        private const string AcsHostName = "accesscontrol.windows.net";
        private const string StsBaseAddress = "localhost/localsts";
        private const string StsPath = "Trust/13/UserName";

        public static void Main(string[] args)
        {
            string stsAddress = string.Format("https://{0}/{1}", StsBaseAddress, StsPath);
            string acsSTSAddress = string.Format("https://{0}.{1}/WRAPv0.8", ServiceNamespace, AcsHostName);

            string samlAssertion = GetSamlAssertion(stsAddress, acsSTSAddress);
            string acsToken = GetACSToken(samlAssertion);

            // create the binding and address to communicate with the service
            WebHttpBinding binding = new WebHttpBinding(WebHttpSecurityMode.None);
            Uri address = new Uri(@"http://localhost/acsexample");

            WebChannelFactory<IACSExample> channelFactory = new WebChannelFactory<IACSExample>(binding, address);

            IACSExample proxy = channelFactory.CreateChannel();

            using (new OperationContextScope(proxy as IContextChannel))
            {
                string authHeaderValue = "WRAPv0.8" + " " + HttpUtility.UrlDecode(acsToken);

                WebOperationContext.Current.OutgoingRequest.Headers.Add("authorization", authHeaderValue);

                // call the service and get a response
                try
                {
                    Console.Write("\tCalling GetMachineName: ");
                    Console.WriteLine("Machine Name is:" + proxy.GetMachineName());

                    Console.Write("\tCalling GetUserDomainName: ");
                    Console.WriteLine("User Domain Name is:" + proxy.GetUserDomainName());

                    Console.Write("\tCalling GetOSVersion: ");
                    Console.WriteLine("OS Version is:" + proxy.GetOSVersion());


                    Console.Write("\tCalling EncodeString: ");
                    Console.WriteLine("Encoded String is:" + Encoding.UTF8.GetString(proxy.EncodeString("Welcome to ProAzure.")));
                }
                catch (MessageSecurityException ex)
                {
                    if (ex.InnerException != null)
                    {
                        WebException wex = ex.InnerException as WebException;
                        if (wex != null)
                        {
                            Console.WriteLine("Error: {0}", wex.Message);
                        }
                    }
                    else
                    {
                        throw;
                    }
                }
            }

            ((IClientChannel)proxy).Close();

            channelFactory.Close();

            Console.ReadLine();
        }

        private static string GetSamlAssertion(string stsAddress, string acsStsAddress)
        {
            WSTrustChannelFactory trustChannelFactory = new WSTrustChannelFactory(
                new WindowsWSTrustBinding(SecurityMode.TransportWithMessageCredential),
                new EndpointAddress(new Uri(stsAddress)));

            trustChannelFactory.TrustVersion = TrustVersion.WSTrust13;

            RequestSecurityToken rst = new RequestSecurityToken(WSTrust13Constants.RequestTypes.Issue, WSTrust13Constants.KeyTypes.Bearer);
            rst.AppliesTo = new EndpointAddress(acsStsAddress);
            rst.TokenType = Microsoft.IdentityModel.Tokens.SecurityTokenTypes.Saml2TokenProfile11;

            WSTrustChannel channel = (WSTrustChannel)trustChannelFactory.CreateChannel();
            GenericXmlSecurityToken token = channel.Issue(rst) as GenericXmlSecurityToken;

            return token.TokenXml.OuterXml;
        }

        private static string GetACSToken(string samlAssertion)
        {
            WebClient tokenClient = new WebClient();
            tokenClient.BaseAddress = string.Format("https://{0}.{1}", ServiceNamespace, AcsHostName);

            NameValueCollection values = new NameValueCollection();
            values.Add("wrap_SAML", samlAssertion);
            values.Add("applies_to", "http://localhost/acsexample");

            byte[] responseBytes = tokenClient.UploadValues("WRAPv0.8", values);
            string response = Encoding.UTF8.GetString(responseBytes);

            return response
               .Split('&')
               .Single(value => value.StartsWith("wrap_token=", StringComparison.OrdinalIgnoreCase))
               .Split('=')[1];
        }
    }
}